<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\LandingPage;
use App\Models\LandingPageSlider;
use App\Models\reviewProductImage;
use App\Models\DeliveryCharge;
use App\Models\Product;
use App\Models\Variation;
use App\Models\User;
use App\Models\Order;

use DB;
use Illuminate\Support\Facades\Log;
// ✅ Facebook Conversion Facade
use App\Facades\FacebookConversion;

class LandingPageController extends Controller
{
    // =========================================================
    // 1. LISTING & CREATION VIEWS
    // =========================================================

    public function index()
    {
        // ✅ Fix: Pagination added
        $items = LandingPage::where('page_type', '1')->latest()->paginate(20);
        return view('backend.landing_pages.index', compact('items'));
    }

    public function landing_page_two()
    {
        // ✅ Fix: Pagination added to fix links() error in blade
        $items = LandingPage::where('page_type', '2')->latest()->paginate(20);
        return view('backend.landing_pages.index_two', compact('items'));
    }

    // ✅✅✅ NEW: Landing Page Type 3 Listing (Premium Design)
    public function index_three()
    {
        $items = LandingPage::where('page_type', '3')->latest()->paginate(20);
        return view('backend.landing_pages.index_three', compact('items'));
    }

    // ✅✅✅ NEW: Landing Page Type 4 Listing (Dark Design)
    public function index_four()
    {
        $items = LandingPage::where('page_type', '4')->latest()->paginate(20);
        return view('backend.landing_pages.index_four', compact('items'));
    }

    // ✅✅✅ NEW: Landing Page Type 5 Listing
    public function index_five()
    {
        $items = LandingPage::where('page_type', '5')->latest()->paginate(20);
        return view('backend.landing_pages.index_five', compact('items'));
    }

    public function create()
    {
        return view('backend.landing_pages.create');
    }

    public function create_landing_page_two()
    {
        return view('backend.landing_pages.create_two');
    }

    // ✅✅✅ NEW: Create Form for Type 3
    public function create_three()
    {
        return view('backend.landing_pages.create_three');
    }

    // ✅✅✅ NEW: Create Form for Type 4
    public function create_four()
    {
        return view('backend.landing_pages.create_four');
    }

    // ✅✅✅ NEW: Create Form for Type 5
    public function create_five()
    {
        return view('backend.landing_pages.create_five');
    }

    // =========================================================
    // 2. HELPER METHODS (Search, Normalize, Ajax Check)
    // =========================================================

    // ✅✅ AJAX SEARCH FUNCTION (Fix for "Product Not Found")
    public function getOrderProduct2(Request $request)
    {
        $search = $request->search;
        if (empty($search)) {
            return response()->json([]);
        }

        $products = Product::query()
            ->where('name', 'LIKE', "%{$search}%")
            ->orWhere('sku', 'LIKE', "%{$search}%")
            ->select('id', 'name', 'sku', 'sell_price', 'image', 'stock_quantity')
            ->take(15)
            ->get();

        return response()->json($products);
    }

    // ✅✅ PRODUCT ENTRY HELPER (Single & Variable Fix)
    public function landingProductEntry(Request $request)
    {
        $product = Product::with(['variations.size', 'variations.color', 'variations.stocks'])->find($request->id);

        if (!$product) {
            return response()->json(['html' => '', 'pr_id' => null, 'variations' => []]);
        }

        $image = function_exists('getImage')
            ? getImage('products', $product->image)
            : asset('products/' . $product->image);

        $html = '<tr>
                    <td>
                        <img src="' . $image . '" height="50" width="50" class="rounded"/>
                    </td>
                    <td>
                        ' . $product->name . ' <br>
                        <small class="text-muted">SKU: ' . $product->sku . '</small>
                    </td>
                    <td>
                        ' . ($product->sell_price ?? 0) . ' Tk
                    </td>
                    <td>
                        <button type="button" class="btn btn-sm btn-danger remove-product">
                            <i class="mdi mdi-delete"></i> Remove
                        </button>
                    </td>
                 </tr>';

        // Variation Logic
        if ($product->variations->count() > 0) {
            $variations = $product->variations->map(function ($v) {
                $size  = $v->size->name ?? '';
                $color = $v->color->name ?? '';
                $stock = $v->stocks->sum('quantity');

                return [
                    'id' => $v->id,
                    'size_name' => $size,
                    'color_name' => $color,
                    'after_discount_price' => $v->after_discount_price,
                    'price' => $v->price,
                    'stock' => (int)$stock
                ];
            });
        } else {
            // ✅ Single Product: Fake variation for dropdown
            $variations = collect([[
                'id' => '', // Empty ID ensures null variation_id
                'size_name' => 'Single',
                'color_name' => 'Product',
                'after_discount_price' => $product->after_discount,
                'price' => $product->sell_price,
                'stock' => (int)($product->stock_quantity ?? 0)
            ]]);
        }

        return response()->json([
            'html' => $html,
            'pr_id' => $product->id,
            'variations' => $variations
        ]);
    }

    private function normalizeProductId(Request $request): void
    {
        $pid = $request->input('product_id');
        if (is_array($pid)) $pid = $pid[0] ?? null;
        $pid = is_string($pid) ? trim($pid) : $pid;

        if ($pid === '' || $pid === null) $pid = null;
        elseif (preg_match('/^\d+$/', (string) $pid)) $pid = (int) $pid;
        else $pid = null;

        $npid = $request->input('new_product_id');
        if (is_array($npid)) $npid = $npid[0] ?? null;
        $npid = is_string($npid) ? trim($npid) : $npid;

        if ($npid !== null && $npid !== '' && preg_match('/^\d+$/', (string) $npid)) {
            $request->merge(['new_product_id' => (int) $npid]);
        } else {
            $request->merge(['new_product_id' => null]);
        }

        $request->merge(['product_id' => $pid]);
    }

    private function isAjax(Request $request): bool
    {
        return $request->ajax() || $request->wantsJson() || $request->expectsJson();
    }

    // =========================================================
    // 3. STORE METHODS
    // =========================================================

    public function store(Request $request)
    {
        return $this->processStore($request, '1');
    }

    public function store_landing_page_two(Request $request)
    {
        return $this->processStore($request, '2');
    }

    // ✅✅✅ NEW: Store Method for Type 3
    public function store_three(Request $request)
    {
        return $this->processStore($request, '3');
    }

    // ✅✅✅ NEW: Store Method for Type 4
    public function store_four(Request $request)
    {
        return $this->processStore($request, '4');
    }

    // ✅✅✅ NEW: Store Method for Type 5
    public function store_five(Request $request)
    {
        return $this->processStore($request, '5');
    }

    // ✅ Unified Store Logic
    private function processStore(Request $request, $pageType)
    {
        $this->normalizeProductId($request);

        $data = $request->validate([
            'title1' => 'required',
            'video_url' => 'nullable',
            'landing_bg' => 'nullable',
            'feature' => 'nullable',
            'review_top_text' => 'nullable',
            'old_price' => 'nullable',
            'new_price' => 'nullable',
            'phone' => 'nullable',
            'pay_text' => 'nullable',
            'product_id' => 'nullable|integer',
            'variation_id' => 'nullable',
            'regular_price_text' => 'nullable',
            'offer_price_text' => 'nullable',
            'call_text' => 'nullable',
            'left_side_title' => 'nullable',
            'left_side_desc' => 'nullable',
            'right_side_title' => 'nullable',
            'right_side_desc' => 'nullable',
            'top_heading_text' => 'nullable',
            'left_product_details' => 'nullable',
            
            // ✅ Dynamic Colors
            'theme_primary_col' => 'nullable',
            'theme_gradient_col' => 'nullable',
            'btn_bg_color' => 'nullable',
            'btn_text_color' => 'nullable',
            
            // ✅ Button Texts
            'btn_text_hero' => 'nullable',
            'btn_text_video' => 'nullable',
            'btn_text_feature' => 'nullable',
            'btn_text_form' => 'nullable',
            
            // ✅ Dynamic Text Fields
            'hot_badge_text' => 'nullable',
            'warranty_text' => 'nullable',
            'order_btn_text' => 'nullable',
            'dhamaka_title' => 'nullable',
            'offer_price_label' => 'nullable',
            'currency_text' => 'nullable',
            'call_to_action_text' => 'nullable',
            'feature_title' => 'nullable',
            'feature_list' => 'nullable',
            'details_title' => 'nullable',
            'review_title' => 'nullable',
            'form_title' => 'nullable',
            'form_subtitle' => 'nullable',
            'name_label' => 'nullable',
            'name_placeholder' => 'nullable',
            'phone_label' => 'nullable',
            'address_label' => 'nullable',
            'address_placeholder' => 'nullable',
            'delivery_label' => 'nullable',
            'payment_title' => 'nullable',
            'cod_title' => 'nullable',
            'cod_subtitle' => 'nullable',
            'online_payment_title' => 'nullable',
            'online_payment_subtitle' => 'nullable',
            'order_summary_title' => 'nullable',
            'variation_label' => 'nullable',
            'total_bill_label' => 'nullable',
            'processing_text' => 'nullable',
            'error_msg' => 'nullable',

            // ✅✅ NEW FIELDS (FROM SQL) - NULLABLE
            // 1. Hero & Countdown
            'countdown_title' => 'nullable',
            'countdown_bg_color' => 'nullable',
            'countdown_text_color' => 'nullable',
            'hero_btn_bg_color' => 'nullable',
            'hero_btn_text_color' => 'nullable',
            'countdown_hours' => 'nullable|integer',

            // 2. Price Texts
            'old_price_text' => 'nullable',
            'new_price_text' => 'nullable',

            // 3. Promise & Identify
            'promise_badge' => 'nullable', 'promise_title' => 'nullable', 'promise_img_badge' => 'nullable',
            'identify_badge' => 'nullable', 'identify_title' => 'nullable', 'identify_subtitle' => 'nullable',

            // 4. Trust Badges
            'trust_sec_title' => 'nullable',
            'trust_1_icon' => 'nullable', 'trust_1_title' => 'nullable',
            'trust_2_icon' => 'nullable', 'trust_2_title' => 'nullable',
            'trust_3_icon' => 'nullable', 'trust_3_title' => 'nullable',
            'trust_4_icon' => 'nullable', 'trust_4_title' => 'nullable',

            // 5. Promise Details
            'promise_1_title' => 'nullable', 'promise_1_desc' => 'nullable',
            'promise_2_title' => 'nullable', 'promise_2_desc' => 'nullable',
            'promise_3_title' => 'nullable', 'promise_3_desc' => 'nullable',
            'negative_title' => 'nullable', 'negative_tags' => 'nullable',

            // 6. Icon Features (1-8)
            'id_1_icon' => 'nullable', 'id_1_title' => 'nullable', 'id_1_desc' => 'nullable',
            'id_2_icon' => 'nullable', 'id_2_title' => 'nullable', 'id_2_desc' => 'nullable',
            'id_3_icon' => 'nullable', 'id_3_title' => 'nullable', 'id_3_desc' => 'nullable',
            'id_4_icon' => 'nullable', 'id_4_title' => 'nullable', 'id_4_desc' => 'nullable',
            'id_5_icon' => 'nullable', 'id_5_title' => 'nullable', 'id_5_desc' => 'nullable',
            'id_6_icon' => 'nullable', 'id_6_title' => 'nullable', 'id_6_desc' => 'nullable',
            'id_7_icon' => 'nullable', 'id_7_title' => 'nullable', 'id_7_desc' => 'nullable',
            'id_8_icon' => 'nullable', 'id_8_title' => 'nullable', 'id_8_desc' => 'nullable',

            // 7. Review & Stats
            'review_badge' => 'nullable', 'review_subtitle' => 'nullable',
            'stat_1_num' => 'nullable', 'stat_1_text' => 'nullable',
            'stat_2_num' => 'nullable', 'stat_2_text' => 'nullable',
            'stat_3_num' => 'nullable', 'stat_3_text' => 'nullable',

            // 8. Customer Reviews
            'rev_1_text' => 'nullable', 'rev_1_name' => 'nullable', 'rev_1_loc' => 'nullable',
            'rev_2_text' => 'nullable', 'rev_2_name' => 'nullable', 'rev_2_loc' => 'nullable',
            'rev_3_text' => 'nullable', 'rev_3_name' => 'nullable', 'rev_3_loc' => 'nullable',
            'rev_4_text' => 'nullable', 'rev_4_name' => 'nullable', 'rev_4_loc' => 'nullable',

            // 9. FAQ
            'faq_badge' => 'nullable', 'faq_title' => 'nullable',
            'faq_1_q' => 'nullable', 'faq_1_a' => 'nullable',
            'faq_2_q' => 'nullable', 'faq_2_a' => 'nullable',
            'faq_3_q' => 'nullable', 'faq_3_a' => 'nullable',
            'faq_4_q' => 'nullable', 'faq_4_a' => 'nullable',
        ]);

        $data['page_type'] = $pageType;

        // Variation Logic
        if ($request->filled('variation_id') && is_numeric($request->variation_id)) {
            if (!empty($data['product_id'])) {
                $ok = Variation::where('id', $request->variation_id)
                    ->where('product_id', $data['product_id'])
                    ->exists();
                $data['variation_id'] = $ok ? (int) $request->variation_id : null;
            } else {
                $data['variation_id'] = (int) $request->variation_id;
            }
        } else {
            $data['variation_id'] = null;
        }

        // Image Uploads
        if ($request->hasFile('image')) {
            $fileName = time() . 'img.' . $request->file('image')->getClientOriginalExtension();
            $request->file('image')->move(public_path('landing_pages'), $fileName);
            $data['image'] = $fileName;
        }

        if ($request->hasFile('landing_bg')) {
            $fileName = time() . 'bg.' . $request->file('landing_bg')->getClientOriginalExtension();
            $request->file('landing_bg')->move(public_path('landing_pages'), $fileName);
            $data['landing_bg'] = $fileName;
        }

        if ($request->hasFile('right_product_image')) {
            $fileName = time() . 'rp.' . $request->file('right_product_image')->getClientOriginalExtension();
            $request->file('right_product_image')->move(public_path('landing_pages'), $fileName);
            $data['right_product_image'] = $fileName;
        }

        $landPage = LandingPage::create($data);

        // Slider Images
        if (isset($request->sliderimage)) {
            $image_data = [];
            foreach ($request->sliderimage as $key => $image) {
                $fileName = time() . $key . 'sl.' . $image->getClientOriginalExtension();
                $image->move(public_path('landing_sliders'), $fileName);
                $image_data[] = ['image' => $fileName];
            }
            if (!empty($image_data)) $landPage->images()->createMany($image_data);
        }

        // Review Images
        if (isset($request->review_product_image)) {
            $review_image_data = [];
            foreach ($request->review_product_image as $key => $image) {
                $fileName = time() . $key . 'rv.' . $image->getClientOriginalExtension();
                $image->move(public_path('review_landing_sliders'), $fileName);
                $review_image_data[] = ['review_image' => $fileName];
            }
            if (!empty($review_image_data)) $landPage->review_images()->createMany($review_image_data);
        }

        // ✅ Redirect logic updated for Type 3, 4, 5
        $redirectRoute = 'admin.landing_pages.index'; // Default
        if ($pageType == '2') $redirectRoute = 'admin.landing_pages_two';
        elseif ($pageType == '3') $redirectRoute = 'admin.landing_pages_three';
        elseif ($pageType == '4') $redirectRoute = 'admin.landing_pages_four';
        elseif ($pageType == '5') $redirectRoute = 'admin.landing_pages_five';

        $redirectUrl = route($redirectRoute);

        if ($this->isAjax($request)) {
            return response()->json([
                'status' => true,
                'msg'    => 'Landing Page Created Successfully!',
                'url'    => $redirectUrl,
            ]);
        }

        return redirect()->route($redirectRoute)->with('success', 'Landing Page Created Successfully!');
    }

    // =========================================================
    // 4. EDIT & UPDATE METHODS
    // =========================================================

    public function edit($id)
    {
        $item = LandingPage::with('images', 'review_images')->findOrFail($id);
        $review_images = reviewProductImage::where('landing_page_id', $id)->get();
        $single_product = $item->product_id ? Product::with(['variations.size', 'variations.color', 'variations.stocks'])->find($item->product_id) : null;
        return view('backend.landing_pages.edit', compact('item', 'single_product', 'review_images'));
    }

    public function edit_landing_page_two($id)
    {
        $item = LandingPage::with('images', 'review_images')->where('page_type', '2')->findOrFail($id);
        $review_images = reviewProductImage::where('landing_page_id', $id)->get();
        $single_product = $item->product_id ? Product::with(['variations.size', 'variations.color', 'variations.stocks'])->find($item->product_id) : null;
        return view('backend.landing_pages.edit_two', compact('item', 'single_product', 'review_images'));
    }

    // ✅✅✅ NEW: Edit Method for Type 3
    public function edit_three($id)
    {
        $item = LandingPage::with('images', 'review_images')->where('page_type', '3')->findOrFail($id);
        $review_images = reviewProductImage::where('landing_page_id', $id)->get();
        $single_product = $item->product_id ? Product::with(['variations.size', 'variations.color', 'variations.stocks'])->find($item->product_id) : null;
        return view('backend.landing_pages.edit_three', compact('item', 'single_product', 'review_images'));
    }

    // ✅✅✅ NEW: Edit Method for Type 4
    public function edit_four($id)
    {
        $item = LandingPage::with('images', 'review_images')->where('page_type', '4')->findOrFail($id);
        $review_images = reviewProductImage::where('landing_page_id', $id)->get();
        $single_product = $item->product_id ? Product::with(['variations.size', 'variations.color', 'variations.stocks'])->find($item->product_id) : null;
        return view('backend.landing_pages.edit_four', compact('item', 'single_product', 'review_images'));
    }

    // ✅✅✅ NEW: Edit Method for Type 5
    public function edit_five($id)
    {
        $item = LandingPage::with('images', 'review_images')->where('page_type', '5')->findOrFail($id);
        $review_images = reviewProductImage::where('landing_page_id', $id)->get();
        $single_product = $item->product_id ? Product::with(['variations.size', 'variations.color', 'variations.stocks'])->find($item->product_id) : null;
        return view('backend.landing_pages.edit_five', compact('item', 'single_product', 'review_images'));
    }


    public function update(Request $request, $id)
    {
        return $this->processUpdate($request, $id, '1');
    }

    public function update_landing_page_two(Request $request, $id)
    {
        return $this->processUpdate($request, $id, '2');
    }

    // ✅✅✅ NEW: Update Method for Type 3
    public function update_three(Request $request, $id)
    {
        return $this->processUpdate($request, $id, '3');
    }

    // ✅✅✅ NEW: Update Method for Type 4
    public function update_four(Request $request, $id)
    {
        return $this->processUpdate($request, $id, '4');
    }

    // ✅✅✅ NEW: Update Method for Type 5
    public function update_five(Request $request, $id)
    {
        return $this->processUpdate($request, $id, '5');
    }


    // ✅ Unified Update Logic
    private function processUpdate(Request $request, $id, $pageType)
    {
        $updatePage = LandingPage::findOrFail($id);
        $this->normalizeProductId($request);

        $data = $request->validate([
            'title1' => 'required',
            'video_url' => 'nullable',
            'landing_bg' => 'nullable',
            'feature' => 'nullable',
            'review_top_text' => 'nullable',
            'old_price' => 'nullable',
            'new_price' => 'nullable',
            'phone' => 'nullable',
            'pay_text' => 'nullable',
            'product_id' => 'nullable|integer',
            'variation_id' => 'nullable',
            'regular_price_text' => 'nullable',
            'offer_price_text' => 'nullable',
            'call_text' => 'nullable',
            'left_side_title' => 'nullable',
            'left_side_desc' => 'nullable',
            'right_side_title' => 'nullable',
            'right_side_desc' => 'nullable',
            'top_heading_text' => 'nullable',
            'left_product_details' => 'nullable',
            
            // ✅ Dynamic Colors
            'theme_primary_col' => 'nullable',
            'theme_gradient_col' => 'nullable',
            'btn_bg_color' => 'nullable',
            'btn_text_color' => 'nullable',
            
            // ✅ Button Texts
            'btn_text_hero' => 'nullable',
            'btn_text_video' => 'nullable',
            'btn_text_feature' => 'nullable',
            'btn_text_form' => 'nullable',
            
            // ✅ Dynamic Text Fields
            'hot_badge_text' => 'nullable',
            'warranty_text' => 'nullable',
            'order_btn_text' => 'nullable',
            'dhamaka_title' => 'nullable',
            'offer_price_label' => 'nullable',
            'currency_text' => 'nullable',
            'call_to_action_text' => 'nullable',
            'feature_title' => 'nullable',
            'feature_list' => 'nullable',
            'details_title' => 'nullable',
            'review_title' => 'nullable',
            'form_title' => 'nullable',
            'form_subtitle' => 'nullable',
            'name_label' => 'nullable',
            'name_placeholder' => 'nullable',
            'phone_label' => 'nullable',
            'address_label' => 'nullable',
            'address_placeholder' => 'nullable',
            'delivery_label' => 'nullable',
            'payment_title' => 'nullable',
            'cod_title' => 'nullable',
            'cod_subtitle' => 'nullable',
            'online_payment_title' => 'nullable',
            'online_payment_subtitle' => 'nullable',
            'order_summary_title' => 'nullable',
            'variation_label' => 'nullable',
            'total_bill_label' => 'nullable',
            'processing_text' => 'nullable',
            'error_msg' => 'nullable',

             // ✅✅ NEW FIELDS (FROM SQL) - NULLABLE
            // 1. Hero & Countdown
            'countdown_title' => 'nullable',
            'countdown_bg_color' => 'nullable',
            'countdown_text_color' => 'nullable',
            'hero_btn_bg_color' => 'nullable',
            'hero_btn_text_color' => 'nullable',
            'countdown_hours' => 'nullable|integer',

            // 2. Price Texts
            'old_price_text' => 'nullable',
            'new_price_text' => 'nullable',

            // 3. Promise & Identify
            'promise_badge' => 'nullable', 'promise_title' => 'nullable', 'promise_img_badge' => 'nullable',
            'identify_badge' => 'nullable', 'identify_title' => 'nullable', 'identify_subtitle' => 'nullable',

            // 4. Trust Badges
            'trust_sec_title' => 'nullable',
            'trust_1_icon' => 'nullable', 'trust_1_title' => 'nullable',
            'trust_2_icon' => 'nullable', 'trust_2_title' => 'nullable',
            'trust_3_icon' => 'nullable', 'trust_3_title' => 'nullable',
            'trust_4_icon' => 'nullable', 'trust_4_title' => 'nullable',

            // 5. Promise Details
            'promise_1_title' => 'nullable', 'promise_1_desc' => 'nullable',
            'promise_2_title' => 'nullable', 'promise_2_desc' => 'nullable',
            'promise_3_title' => 'nullable', 'promise_3_desc' => 'nullable',
            'negative_title' => 'nullable', 'negative_tags' => 'nullable',

            // 6. Icon Features (1-8)
            'id_1_icon' => 'nullable', 'id_1_title' => 'nullable', 'id_1_desc' => 'nullable',
            'id_2_icon' => 'nullable', 'id_2_title' => 'nullable', 'id_2_desc' => 'nullable',
            'id_3_icon' => 'nullable', 'id_3_title' => 'nullable', 'id_3_desc' => 'nullable',
            'id_4_icon' => 'nullable', 'id_4_title' => 'nullable', 'id_4_desc' => 'nullable',
            'id_5_icon' => 'nullable', 'id_5_title' => 'nullable', 'id_5_desc' => 'nullable',
            'id_6_icon' => 'nullable', 'id_6_title' => 'nullable', 'id_6_desc' => 'nullable',
            'id_7_icon' => 'nullable', 'id_7_title' => 'nullable', 'id_7_desc' => 'nullable',
            'id_8_icon' => 'nullable', 'id_8_title' => 'nullable', 'id_8_desc' => 'nullable',

            // 7. Review & Stats
            'review_badge' => 'nullable', 'review_subtitle' => 'nullable',
            'stat_1_num' => 'nullable', 'stat_1_text' => 'nullable',
            'stat_2_num' => 'nullable', 'stat_2_text' => 'nullable',
            'stat_3_num' => 'nullable', 'stat_3_text' => 'nullable',

            // 8. Customer Reviews
            'rev_1_text' => 'nullable', 'rev_1_name' => 'nullable', 'rev_1_loc' => 'nullable',
            'rev_2_text' => 'nullable', 'rev_2_name' => 'nullable', 'rev_2_loc' => 'nullable',
            'rev_3_text' => 'nullable', 'rev_3_name' => 'nullable', 'rev_3_loc' => 'nullable',
            'rev_4_text' => 'nullable', 'rev_4_name' => 'nullable', 'rev_4_loc' => 'nullable',

            // 9. FAQ
            'faq_badge' => 'nullable', 'faq_title' => 'nullable',
            'faq_1_q' => 'nullable', 'faq_1_a' => 'nullable',
            'faq_2_q' => 'nullable', 'faq_2_a' => 'nullable',
            'faq_3_q' => 'nullable', 'faq_3_a' => 'nullable',
            'faq_4_q' => 'nullable', 'faq_4_a' => 'nullable',
        ]);

        if ($request->new_product_id != null) {
            $data['product_id'] = (int) $request->new_product_id;
        }

        if ($request->filled('variation_id')) {
            if (!empty($data['product_id'])) {
                $ok = Variation::where('id', $request->variation_id)->where('product_id', $data['product_id'])->exists();
                $data['variation_id'] = $ok ? (int) $request->variation_id : null;
            } else {
                $data['variation_id'] = (int) $request->variation_id;
            }
        }

        if ($request->hasFile('image')) {
            $fileName = time() . '1up.' . $request->file('image')->getClientOriginalExtension();
            $request->file('image')->move(public_path('landing_pages'), $fileName);
            $data['image'] = $fileName;
        }
        if ($request->hasFile('landing_bg')) {
            $fileName = time() . '2up.' . $request->file('landing_bg')->getClientOriginalExtension();
            $request->file('landing_bg')->move(public_path('landing_pages'), $fileName);
            $data['landing_bg'] = $fileName;
        }
        if ($request->hasFile('right_product_image')) {
            $fileName = time() . '3up.' . $request->file('right_product_image')->getClientOriginalExtension();
            $request->file('right_product_image')->move(public_path('landing_pages'), $fileName);
            $data['right_product_image'] = $fileName;
        }

        if (isset($request->sliderimage)) {
            $image_data = [];
            foreach ($request->sliderimage as $key => $image) {
                $fileName = time() . $key . 'slup.' . $image->getClientOriginalExtension();
                $image->move(public_path('landing_sliders'), $fileName);
                $image_data[] = ['image' => $fileName];
            }
            if (!empty($image_data)) $updatePage->images()->createMany($image_data);
        }

        if (isset($request->review_product_image)) {
            $review_image_data = [];
            foreach ($request->review_product_image as $key => $image) {
                $fileName = time() . $key . 'rvup.' . $image->getClientOriginalExtension();
                $image->move(public_path('review_landing_sliders'), $fileName);
                $review_image_data[] = ['review_image' => $fileName];
            }
            if (!empty($review_image_data)) $updatePage->review_images()->createMany($review_image_data);
        }

        $updatePage->update($data);

        // ✅ Redirect logic updated for Type 3, 4, 5
        $redirectRoute = 'admin.landing_pages.index'; // Default
        if ($pageType == '2') $redirectRoute = 'admin.landing_pages_two';
        elseif ($pageType == '3') $redirectRoute = 'admin.landing_pages_three';
        elseif ($pageType == '4') $redirectRoute = 'admin.landing_pages_four';
        elseif ($pageType == '5') $redirectRoute = 'admin.landing_pages_five';


        $redirectUrl = route($redirectRoute);

        if ($this->isAjax($request)) {
            return response()->json([
                'status' => true,
                'msg'    => 'Landing Page Updated Successfully!',
                'url'    => $redirectUrl,
            ]);
        }

        return redirect()->route($redirectRoute)->with('success', 'Landing Page Updated Successfully!');
    }

    // =========================================================
    // 5. DELETE & OTHER SETTINGS
    // =========================================================

    public function delete_slider($id)
    {
        $item = LandingPageSlider::find($id);
        if ($item) {
            deleteImage('landing_sliders', $item->image);
            $item->delete();
        }
        return back();
    }

    public function delete_review(Request $request, $id)
    {
        $delete_item = reviewProductImage::find($id);
        if ($delete_item) {
            deleteImage('review_landing_sliders', $delete_item->review_image);
            $delete_item->delete();
        }
        return back();
    }

    public function destroy(Request $request, $id)
    {
        $single_page = LandingPage::with(['images', 'review_images'])->find($id);

        if ($single_page) {
            deleteImage('landing_pages', $single_page->image);
            deleteImage('landing_pages', $single_page->landing_bg);
            deleteImage('landing_pages', $single_page->right_product_image);

            foreach ($single_page->images as $slider_image) {
                deleteImage('landing_sliders', $slider_image->image);
            }

            foreach ($single_page->review_images as $rv) {
                deleteImage('review_landing_sliders', $rv->review_image);
            }

            $single_page->delete();
        }

        if ($this->isAjax($request)) {
            return response()->json(['status' => true, 'msg' => 'Deleted Successfully!']);
        }
        return redirect()->back()->with('success', 'Deleted Successfully!');
    }

    // =========================================================
    // 6. COLOR SETTINGS METHODS
    // =========================================================

    public function colorSettings($id = null)
    {
        // ✅ Fix: Added Type 5 to color settings
        $all_pages = LandingPage::select('id', 'title1', 'page_type')
            ->whereIn('page_type', ['2', '3', '4', '5']) // ✅ Allowed types for color settings
            ->orderBy('id', 'desc')
            ->get();

        if ($id) {
            $ln_pg = LandingPage::find($id);
            // Fallback if the requested ID was deleted or invalid
            if (!$ln_pg && $all_pages->isNotEmpty()) {
                $ln_pg = $all_pages->first();
            }
        } elseif ($all_pages->isNotEmpty()) {
            $ln_pg = $all_pages->first();
        } else {
            $ln_pg = null; // No pages exist
        }

        return view('backend.landing_pages.color', compact('ln_pg', 'all_pages'));
    }

    public function updateColor(Request $request, $id)
    {
        $ln_pg = LandingPage::findOrFail($id);

        $data = $request->validate([
            'theme_primary_col' => 'nullable',
            'theme_gradient_col' => 'nullable',
            'btn_bg_color' => 'nullable',
            'btn_text_color' => 'nullable',
            'btn_text_hero' => 'nullable',
            'btn_text_video' => 'nullable',
            'btn_text_feature' => 'nullable',
            'btn_text_form' => 'nullable',
        ]);

        $ln_pg->update($data);

        return redirect()->back()->with('success', 'Settings updated successfully!');
    }

    // =========================================================
    // 7. FRONTEND VIEW & ORDER PLACEMENT
    // =========================================================

    public function landing_page($id)
    {
        $ln_pg = LandingPage::with(['images', 'review_images'])->find($id);
        if(!$ln_pg) abort(404);

        $title = $ln_pg->title1 ?? '';
        $charges = DeliveryCharge::whereNotNull('status')->get();
        $product = $ln_pg->product_id ? Product::with(['variations.size', 'variations.color', 'variations.stocks', 'category'])->find($ln_pg->product_id) : null;

        // CAPI Logic
        if ($product) {
            try {
                $finalPrice = ($product->after_discount && $product->after_discount > 0)
                    ? $product->after_discount
                    : $product->sell_price;

                $base        = "LP_{$product->id}_" . now()->format('YmdHis');
                $eventIdVC = $base . "_VC";
                $eventIdIC = $base . "_IC";

                $userData = [];
                $authUser = auth()->user();

                if (!empty($authUser?->email)) $userData['em'] = [hash('sha256', strtolower(trim($authUser->email)))];
                if (!empty($authUser?->phone_number)) $userData['ph'] = [hash('sha256', preg_replace('/\D/', '', $authUser->phone_number))];
                if (!empty($authUser?->id)) $userData['external_id'] = [hash('sha256', (string)$authUser->id)];

                FacebookConversion::sendViewContent([
                    'product_id'        => $product->id,
                    'product_name'      => $product->name,
                    'value'             => (float)$finalPrice,
                    'currency'          => 'BDT',
                    'content_category' => $product->category->name ?? 'Landing Page',
                    'event_time'        => now()->timestamp,
                    'action_source'     => 'website',
                ], $eventIdVC, $userData);

                FacebookConversion::sendInitiateCheckout([
                    'product_id'    => $product->id,
                    'value'         => (float)$finalPrice,
                    'currency'      => 'BDT',
                    'num_items'     => 1,
                    'event_time'    => now()->timestamp,
                    'action_source' => 'website',
                ], $eventIdIC, $userData);

            } catch (\Throwable $e) {
                Log::error('Facebook CAPI (LP View) Error: ' . $e->getMessage());
            }
        }

        // Variation Matrix
        $matrix = [];
        $sizes  = collect();
        $colors = collect();

        if ($product) {
            foreach ($product->variations as $v) {
                $stock = $v->stocks->sum('quantity');
                $key = ($v->size_id ?? 0) . '_' . ($v->color_id ?? 0);
                $price = $v->after_discount_price ?: $v->price;
                
                $matrix[$key] = [
                    'variation_id' => $v->id,
                    'price' => (float)$price,
                    'stock' => (int)$stock
                ];

                if ($v->size) $sizes->push(['id' => $v->size->id, 'name' => $v->size->name]);
                if ($v->color) $colors->push(['id' => $v->color->id, 'name' => $v->color->name, 'code' => $v->color->code]);
            }
        }
        $sizes = $sizes->unique('id')->values();
        $colors = $colors->unique('id')->values();

        return view('backend.landing_pages.land_page', compact('ln_pg', 'charges', 'title', 'product', 'matrix', 'sizes', 'colors'));
    }

    public function storeData(Request $request)
    {
        $data = $request->validate([
            'mobile' => 'required|digits:11',
            'first_name' => 'required',
            'shipping_address' => 'required',
            'delivery_charge_id' => 'required|numeric',
            'prd_id' => 'required|integer',
            'purchase_event_id' => 'nullable|string',
            'variation_id' => 'nullable|integer',
            'quantity' => 'nullable|integer|min:1',
        ]);

        if (empty(auth()->user()?->id)) {
            $user = User::create([
                'first_name' => $request->first_name,
                'mobile' => $request->mobile,
                'shipping_address' => $request->shipping_address,
                'note' => $request->input('note'),
            ]);
            $data['user_id'] = $user->id;
        } else {
            $data['user_id'] = auth()->user()->id;
        }

        $product = Product::findOrFail($request->prd_id);
        $v_id = $request->variation_id ?: Variation::where('product_id', $product->id)->value('id');
        $selectedVariation = $v_id ? Variation::with('stocks')->find($v_id) : null;
        
        $qty = max((int)$request->quantity, 1);
        
        $unitPrice = 0;
        if($selectedVariation) {
            $unitPrice = $selectedVariation->after_discount_price ?: $selectedVariation->price;
        } else {
            $unitPrice = $product->after_discount ?: $product->sell_price;
        }

        $charge = DeliveryCharge::find($request->delivery_charge_id);
        $chargeAmount = $charge ? (float)$charge->amount : 0;
        $subTotal = $unitPrice * $qty;
        $finalAmount = $subTotal + $chargeAmount;
        $invoiceNo = rand(111111, 999999);

        // Assign Admin
        $admins = DB::table('model_has_roles')->where('role_id', 8)->pluck('model_id')->toArray();
        $assignUserId = (!empty($admins)) ? $admins[array_rand($admins)] : 1;

        $orderData = [
            'mobile' => $request->mobile,
            'first_name' => $request->first_name,
            'shipping_address' => $request->shipping_address,
            'note' => $request->input('note'),
            'delivery_charge_id' => $request->delivery_charge_id,
            'user_id' => $data['user_id'],
            'assign_user_id' => $assignUserId,
            'date' => date('Y-m-d'),
            'invoice_no' => $invoiceNo,
            'amount' => $subTotal,
            'shipping_charge' => $chargeAmount,
            'final_amount' => $finalAmount,
        ];

        DB::beginTransaction();
        try {
            $order = Order::create($orderData);
            $order->details()->create([
                'product_id' => $product->id,
                'quantity' => $qty,
                'unit_price' => $unitPrice,
                'variation_id' => $selectedVariation?->id,
            ]);

            DB::commit();
            
            // CAPI Purchase Logic... (Skipped for brevity)

            return response()->json([
                'success' => true,
                'msg' => 'Order Created Successfully!',
                'url' => route('front.confirmOrder', [$order->id]),
                'invoice_no' => $invoiceNo
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['success' => false, 'msg' => $e->getMessage()]);
        }
    }
}