<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\OrderDetails;
use App\Models\User;
use App\Models\Courier;
use App\Models\Product;
use Illuminate\Support\Facades\DB;
use App\Exports\OrderReportExport;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon;
use Symfony\Component\HttpFoundation\StreamedResponse;

class ReportController extends Controller
{
    /**
     * ✅ 1. SALES REPORT (UPDATED)
     * SSL Payment & Product Wise Sales Added
     */
    public function salesReport(Request $request)
    {
        // ১. ফিল্টার সেটআপ
        $startDate = $request->start_date ? Carbon::parse($request->start_date)->startOfDay() : Carbon::now()->startOfMonth();
        $endDate   = $request->end_date ? Carbon::parse($request->end_date)->endOfDay() : Carbon::now()->endOfDay();
        $status    = $request->status ?? 'all';

        // ২. কুয়েরি তৈরি
        $query = Order::query();

        // তারিখ ফিল্টার
        $query->whereBetween('created_at', [$startDate, $endDate]);

        // স্ট্যাটাস ফিল্টার
        if ($status != 'all') {
            $query->where('status', $status);
        } else {
            // Cancelled বা Returned অর্ডার Sales হিসেবে কাউন্ট না করার জন্য
            $query->whereNotIn('status', ['canceled', 'cancelled', 'returned']);
        }

        // ৩. রিলেশনশিপ লোড (details এবং product)
        $orders = $query->with(['details.product'])
                        ->orderBy('created_at', 'desc')
                        ->get();

        // ৪. ক্যালকুলেশন ভেরিয়েবল
        $totalOrders      = $orders->count();
        $totalSales       = 0; // কাস্টমারের দেওয়া মোট টাকা (Final Amount)
        $totalDelivery    = 0;
        $totalProductCost = 0; // কেনা দাম
        $totalItemsSold   = 0;

        // ✅ NEW ADDED: অনলাইন পেইড এমাউন্ট (SSLCommerz)
        // লজিক: যাদের transaction_id আছে, তারাই অনলাইনে পেমেন্ট করেছে
        $totalOnlinePaid = $orders->whereNotNull('transaction_id')->sum('final_amount');

        // ৫. লুপ চালিয়ে হিসাব বের করা
        foreach ($orders as $order) {
            $currentOrderSales = $order->final_amount; 
            $totalSales += $currentOrderSales;

            $delivery = $order->shipping_charge ?? 0;
            $totalDelivery += $delivery;

            // কেনা দাম বের করা (Profit Calculation)
            foreach ($order->details as $item) {
                $qty = $item->quantity;
                $totalItemsSold += $qty;

                // products টেবিলে purchase_price থাকতে হবে
                $purchasePrice = $item->product ? ($item->product->purchase_price ?? 0) : 0;
                
                $totalProductCost += ($purchasePrice * $qty);
            }
        }

        // ৬. প্রফিট ক্যালকুলেশন
        // Net Sales = (Total Sales - Delivery Charge)
        $netSales = $totalSales - $totalDelivery;

        // Gross Profit = Net Sales - Total Product Cost
        $grossProfit = $netSales - $totalProductCost;

        // ✅ NEW ADDED: Product Wise Sales List (Top Selling Products)
        // এই লিস্টটি ব্লেড ফাইলে লুপ করে দেখানো হবে
        $soldProducts = OrderDetails::with('product')
            ->whereIn('order_id', $orders->pluck('id')) // শুধুমাত্র ফিল্টার করা অর্ডারের প্রোডাক্ট
            ->select('product_id', DB::raw('SUM(quantity) as total_qty'), DB::raw('SUM(unit_price * quantity) as total_amount'))
            ->groupBy('product_id')
            ->orderBy('total_qty', 'desc') // বেশি বিক্রি হওয়া প্রোডাক্ট আগে থাকবে
            ->get();

        // ৭. চার্ট ডাটা তৈরি (Daily Sales Chart)
        $dailyStats = $orders->groupBy(function($date) {
            return Carbon::parse($date->created_at)->format('Y-m-d');
        })->map(function ($row) {
            return $row->sum('final_amount');
        });

        $chartLabels = [];
        $chartData   = [];
        
        // চার্টে যাতে গ্যাপ না থাকে তাই লুপ চালানো হচ্ছে
        $period = \Carbon\CarbonPeriod::create($startDate, $endDate);
        foreach ($period as $date) {
            $dateString = $date->format('Y-m-d');
            $chartLabels[] = $date->format('d M');
            $chartData[] = isset($dailyStats[$dateString]) ? $dailyStats[$dateString] : 0;
        }

        return view('backend.reports.sales', compact(
            'orders', 'totalOrders', 'totalSales', 'totalDelivery',
            'totalProductCost', 'grossProfit', 'netSales',
            'startDate', 'endDate', 'status', 'chartLabels', 'chartData',
            'totalOnlinePaid', 'soldProducts' // ✅ নতুন ভেরিয়েবল পাঠানো হলো
        ));
    }

    /**
     * ✅ EXPORT SALES REPORT (CSV)
     * আপডেটেড: পেমেন্ট মেথড এবং ট্রানজেকশন আইডি যোগ করা হয়েছে
     */
    public function exportSales(Request $request)
    {
        // ১. ফিল্টার অনুযায়ী ডাটা আনা
        $startDate = $request->start_date ? Carbon::parse($request->start_date)->startOfDay() : Carbon::now()->startOfMonth();
        $endDate   = $request->end_date ? Carbon::parse($request->end_date)->endOfDay() : Carbon::now()->endOfDay();
        $status    = $request->status ?? 'all';

        $query = Order::query()->with('details.product');

        // Date Filter
        $query->whereBetween('created_at', [$startDate, $endDate]);

        // Status Filter
        if ($status != 'all') {
            $query->where('status', $status);
        } else {
            $query->whereNotIn('status', ['canceled', 'cancelled', 'returned']);
        }

        $orders = $query->orderBy('created_at', 'desc')->get();

        // ২. CSV জেনারেট করা
        $response = new StreamedResponse(function() use ($orders) {
            $handle = fopen('php://output', 'w');

            // CSV Header (কলামের নাম)
            fputcsv($handle, [
                'Invoice No',
                'Customer Name',
                'Mobile',
                'Date',
                'Status',
                'Payment Method', // ✅ Added
                'Transaction ID', // ✅ Added
                'Items Qty',
                'Total Amount (Tk)',
                'Delivery Charge',
                'Product Cost',
                'Profit'
            ]);

            // Data Rows
            foreach ($orders as $order) {
                // Cost & Profit Calculation
                $orderCost = 0;
                foreach($order->details as $detail) {
                    $pp = $detail->product ? ($detail->product->purchase_price ?? 0) : 0;
                    $orderCost += ($pp * $detail->quantity);
                }
                $delivery = $order->shipping_charge ?? 0;
                $netSales = $order->final_amount - $delivery;
                $profit   = $netSales - $orderCost;

                // ✅ Payment Logic Check
                $payMethod = $order->transaction_id ? 'Online/SSL' : 'COD';
                $trxId     = $order->transaction_id ?? 'N/A';

                // Write row to csv
                fputcsv($handle, [
                    $order->invoice_no,
                    $order->first_name . ' ' . $order->last_name,
                    $order->mobile,
                    $order->created_at->format('d M, Y h:i A'),
                    ucfirst($order->status),
                    $payMethod, // ✅ Added Value
                    $trxId,     // ✅ Added Value
                    $order->details->sum('quantity'),
                    $order->final_amount,
                    $delivery,
                    $orderCost,
                    $profit
                ]);
            }

            fclose($handle);
        });

        // ৩. ফাইল ডাউনলোড হেডার সেট করা
        $response->headers->set('Content-Type', 'text/csv');
        $response->headers->set('Content-Disposition', 'attachment; filename="sales-report-'.date('Y-m-d').'.csv"');

        return $response;
    }

    /**
     * ✅ 2. ORDER REPORT
     */
    public function orderReport()
    {
        $details = OrderDetails::with(['order', 'product', 'variation'])
          ->whereHas('order', function($q){
            $q->whereNotNull('invoice_no');
          })          
          ->whereHas('product', function($q){
            $q->whereNotNull('name');
          })
          ->latest()->paginate(100);
        
        // Admin & Worker roles only
        $users = User::with("roles")->whereHas("roles", function($q) {
                    $q->whereIn("name", ["admin", "worker"]);
                })->get();
 
        $couriers = Courier::all();
        
        return view('backend.reports.order', compact('details', 'users', 'couriers'));
    }
    
    public function filterOrder(Request $request)
    {      
       $details = OrderDetails::join('orders as o', 'order_details.order_id', 'o.id')
                                ->join('products as p', 'order_details.product_id', 'p.id')
                                ->join('variations as v', 'order_details.variation_id', 'v.id')
                                ->select('o.*', 'order_details.*', 'p.*', 'v.*')
                                ->where(function($query){
                                   if(!empty(request()->status))
                                   {
                                      $query->where('o.status', request()->status);
                                   }  
                                   
                                   if(!empty(request()->input('query')))
                                   {
                                      $query->where('o.invoice_no', 'like', '%'.request()->input('query').'%')
                                                  ->orWhere('p.name', 'like', '%'.request()->input('query').'%');
                                   }        
                                   
                                   if(!empty(request()->from && request()->to))
                                   {
                                       $query->whereBetween('o.date', [request()->from, request()->to]);
                                   }        
                                   
                                   if(!empty(request()->assign))
                                   {
                                      $query->where('o.assign_user_id', request()->assign);
                                   }                                     
                                   
                                   if(!empty(request()->courier))
                                   {
                                      $query->where('o.courier_id', request()->courier);
                                   }
                                })
                                ->paginate(100)
                                ->appends($request->all());        
  
        
        $users = User::with("roles")->whereHas("roles", function($q) {
                    $q->whereIn("name", ["admin", "worker"]);
                })->get();
                
        $couriers = Courier::all();
        
        return view('backend.reports.order', compact('details', 'users', 'couriers'));        
       
    }

    public function exportOrderReport()
    {
        $details = OrderDetails::join('orders as o', 'order_details.order_id', 'o.id')
                                ->join('products as p', 'order_details.product_id', 'p.id')
                                ->join('variations as v', 'order_details.variation_id', 'v.id')
                                ->select('o.*', 'order_details.*', 'p.*', 'v.*')
                                ->where(function($query){
                                   if(!empty(request()->status)) {
                                      $query->where('o.status', request()->status);
                                   }  
                                   if(!empty(request()->input('query'))) {
                                      $query->where('o.invoice_no', 'like', '%'.request()->input('query').'%')
                                                  ->orWhere('p.name', 'like', '%'.request()->input('query').'%');
                                   }        
                                   if(!empty(request()->from && request()->to)) {
                                       $query->whereBetween('o.date', [request()->from, request()->to]);
                                   }        
                                   if(!empty(request()->assign)) {
                                      $query->where('o.assign_user_id', request()->assign);
                                   }                                     
                                   if(!empty(request()->courier)) {
                                      $query->where('o.courier_id', request()->courier);
                                   }
                                })
                                ->orderBy('order_details.created_at', 'desc')->get();
        
            return Excel::download(new OrderReportExport($details), 'order_report.xlsx');
    }

    /**
     * ✅ 3. PRODUCT REPORT
     */
    public function productReport()
    {
        $details = OrderDetails::Leftjoin("products as p", "order_details.product_id","p.id")  
                              ->Leftjoin("orders as o","o.id","order_details.order_id")              
                              ->select("p.id","p.name","order_details.unit_price",DB::raw("SUM(quantity) as total_qty"))
                              ->groupBy('p.id','p.name','order_details.unit_price')    
                              ->get();

        $users = User::with("roles")->whereHas("roles", function($q) {
                    $q->whereIn("name", ["admin", "worker"]);
                })->get();
 
        $couriers = Courier::all();
        
        return view('backend.reports.product', compact('details', 'users', 'couriers'));
    }
    
    public function filterProduct(Request $request){  
      
      $details = OrderDetails::Leftjoin("products as p", "order_details.product_id","p.id")  
                              ->Leftjoin("orders as o","o.id","order_details.order_id")              
                              ->select("p.id","p.name","order_details.unit_price",DB::raw("SUM(quantity) as total_qty"))
                               ->where(function($query){
                                 if(!empty(request()->status)) {
                                   $query->where('o.status', request()->status);
                                 }
                                 if(!empty(request()->from && request()->to)) {
                                      $query->whereBetween('o.date', [request()->from, request()->to]);
                                 }        
                                 if(!empty(request()->assign)) {
                                      $query->where('o.checked_by', request()->assign);
                                 }                                     
                                 if(!empty(request()->courier)) {
                                      $query->where('o.courier_id', request()->courier);
                                 }
                               })
                               ->groupBy('p.id','p.name','order_details.unit_price')        
                              ->get();    
        
        $users = User::with("roles")->whereHas("roles", function($q) {
                    $q->whereIn("name", ["admin", "worker"]);
                })->get();
 
        $couriers = Courier::all();
        
        return view('backend.reports.product', compact('details', 'users', 'couriers'));
    }

    /**
     * ✅ 4. USER REPORT
     */
    public function userReport(Request $request){
        if ($request->ajax()) {
            
            $startDate = $request->startDate;
            $endDate = $request->endDate;
            $assignUser = $request->assignUser;
            
            $query = Order::leftJoin('users', 'orders.assign_user_id', '=', 'users.id')
            ->select(
                'users.first_name as assign_user_name',
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(CASE WHEN orders.status = "pending" THEN 1 ELSE 0 END) as pending_orders'),
                DB::raw('SUM(CASE WHEN orders.status = "processing" THEN 1 ELSE 0 END) as processing_orders'),
                DB::raw('SUM(CASE WHEN orders.status = "courier" THEN 1 ELSE 0 END) as courier_orders'),
                DB::raw('SUM(CASE WHEN orders.status = "courier_complete" THEN 1 ELSE 0 END) as courier_complete_orders'),
                DB::raw('SUM(CASE WHEN orders.status = "on_hold" THEN 1 ELSE 0 END) as on_hold_orders'),
                DB::raw('SUM(CASE WHEN orders.status = "complete" THEN 1 ELSE 0 END) as complete_orders')
            )
            ->where('orders.assign_user_id', $assignUser);

            if (!empty($startDate)) {
                $query->where('orders.created_at', '>=', $startDate);
            }
            
            if (!empty($startDate) && !empty($endDate)) {
                $query->whereBetween('orders.created_at', [$startDate, $endDate]);
            }
            
            $query->groupBy('users.first_name');
            
            $items = $query->paginate(20);
            
            $html=view('backend.reports.getUserData',compact('items'))->render();
            return response()->json(['success'=>true,'html'=>$html]);
        }
        
        $users = User::with("roles")->whereHas("roles", function($q) {
                    $q->whereIn("name", ["admin", "worker"]);
                })->get();
                
        return view('backend.reports.user',compact('users'));
    }
}