<?php
/**
 * License Domain Guard
 * Location: vendor/composer/Support/ClassVersionLoader.php
 */

use DateTime;
use DateTimeZone;

/**
 * ------------------------------------------------------------------
 * 1. Daily lock (per day, per server)
 * ------------------------------------------------------------------
 */
$dt = new DateTime('now', new DateTimeZone('Asia/Dhaka'));
$lockFile = dirname(__DIR__, 3) . '/storage/framework/cache/.license_lock_' . $dt->format('Ymd');

// CLI / already checked today
if (php_sapi_name() === 'cli' || file_exists($lockFile)) {
    return;
}

/**
 * ------------------------------------------------------------------
 * 2. Load license settings
 * ------------------------------------------------------------------
 */
$settingsFile = dirname(__DIR__, 3) . '/storage/app/cache/settings.json';

if (!file_exists($settingsFile)) {
    http_response_code(403);
    exit('License settings missing.');
}

$data = json_decode(file_get_contents($settingsFile), true);

if (!is_array($data) || !isset($data['domains']) || !is_array($data['domains'])) {
    http_response_code(403);
    exit('Invalid license data.');
}

/**
 * ------------------------------------------------------------------
 * 3. Normalize allowed domains
 * ------------------------------------------------------------------
 */
$allowedDomains = array_map(function ($url) {
    $host = parse_url($url, PHP_URL_HOST);
    return strtolower($host ?: $url);
}, $data['domains']);

$currentHost = strtolower($_SERVER['HTTP_HOST'] ?? '');

/**
 * ------------------------------------------------------------------
 * 4. Unauthorized → Show premium UI
 * ------------------------------------------------------------------
 */
if (!in_array($currentHost, $allowedDomains, true)) {

    http_response_code(403);

    // ✅ FIX: brand was missing in your code
    $brand        = $data['brand'] ?? 'License Protection';
    $supportEmail = $data['support_email'] ?? 'nerobpikash@gmail.com';
    $docsUrl      = $data['docs_url'] ?? 'https://wa.me/8801627212113';

    $safeHost  = htmlspecialchars($currentHost ?: 'unknown', ENT_QUOTES, 'UTF-8');
    $safeBrand = htmlspecialchars($brand, ENT_QUOTES, 'UTF-8');
    $safeEmail = htmlspecialchars($supportEmail, ENT_QUOTES, 'UTF-8');
    $safeDocs  = htmlspecialchars($docsUrl, ENT_QUOTES, 'UTF-8');

    $requestId = substr(hash('sha256', $safeHost . date('Y-m-d')), 0, 12);

    exit(<<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>403 • Unauthorized Domain</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<style>
:root{
  --bg1:#0b1220;
  --bg2:#0f1b33;
  --card:rgba(255,255,255,.06);
  --border:rgba(255,255,255,.14);
  --text:#e8eefc;
  --muted:rgba(232,238,252,.65);
  --accent1:#6ee7ff;
  --accent2:#a78bfa;
  --danger:#ff5d5d;
  --radius:22px;
  --shadow:0 22px 70px rgba(0,0,0,.45);
}
*{box-sizing:border-box}
body{
  margin:0; min-height:100vh; display:grid; place-items:center;
  font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial;
  color:var(--text);
  background:
    radial-gradient(900px 600px at 15% 15%, rgba(110,231,255,.18), transparent 55%),
    radial-gradient(800px 560px at 85% 20%, rgba(167,139,250,.18), transparent 55%),
    linear-gradient(160deg,var(--bg1),var(--bg2));
  padding:24px;
}
.card{
  width:min(960px,100%);
  background:linear-gradient(180deg,var(--card),rgba(255,255,255,.04));
  border:1px solid var(--border);
  border-radius:var(--radius);
  box-shadow:var(--shadow);
  overflow:hidden;
}
.header{
  padding:18px 22px;
  display:flex; justify-content:space-between; align-items:center;
  border-bottom:1px solid var(--border);
}
.badge{display:flex; align-items:center; gap:10px; font-weight:800;}
.dot{
  width:10px; height:10px; border-radius:50%;
  background:var(--danger);
  box-shadow:0 0 0 6px rgba(255,93,93,.18);
}
.brand{font-weight:700; color:var(--muted);}
.content{
  padding:28px;
  display:grid;
  grid-template-columns:1.2fr .8fr;
  gap:22px;
}
@media(max-width:860px){ .content{grid-template-columns:1fr} }
h1{margin:0 0 10px; font-size:34px;}
p{margin:0 0 14px; color:var(--muted); line-height:1.6;}
.pills{display:flex; flex-wrap:wrap; gap:10px; margin:14px 0;}
.pill{
  padding:8px 12px;
  border-radius:999px;
  background:rgba(255,255,255,.06);
  border:1px solid var(--border);
  font-size:13px;
}
.actions{display:flex; gap:12px; flex-wrap:wrap; margin-top:16px;}
.btn{
  padding:12px 16px;
  border-radius:14px;
  text-decoration:none;
  font-weight:800;
  border:1px solid var(--border);
  background:rgba(255,255,255,.06);
  color:var(--text);
}
.btn.primary{
  background:linear-gradient(135deg,var(--accent1),var(--accent2));
  color:#0b1220;
  border:none;
}
.side{
  background:rgba(255,255,255,.05);
  border:1px solid var(--border);
  border-radius:18px;
  padding:18px;
}
.side h3{
  margin:0 0 12px;
  font-size:13px;
  text-transform:uppercase;
  letter-spacing:.08em;
  color:var(--muted);
}
.kv{
  display:grid;
  grid-template-columns:120px 1fr;
  gap:10px;
  font-size:14px;
}
.footer{
  padding:14px 18px;
  border-top:1px solid var(--border);
  display:flex;
  justify-content:space-between;
  font-size:12px;
  color:var(--muted);
}
code{
  background:rgba(0,0,0,.35);
  padding:4px 8px;
  border-radius:999px;
  font-size:12px;
}
</style>
</head>
<body>

<div class="card">

  <div class="header">
    <div class="badge">
      <span class="dot"></span> 403 • Unauthorized Domain
    </div>
    <div class="brand">$safeBrand</div>
  </div>

  <div class="content">

    <div>
      <h1>Domain not licensed</h1>
      <p>
        This installation is running on <b>$safeHost</b>, but this domain is not
        registered in our license configuration.
      </p>

      <div class="pills">
        <div class="pill">Request ID: <code>$requestId</code></div>
      </div>

      <div class="actions">
        <a class="btn primary"
           href="mailto:$safeEmail?subject=License%20Activation%20$requestId&body=Domain:%20$safeHost%0ARequest%20ID:%20$requestId">
           Contact Support
        </a>
        <a class="btn" href="$safeDocs">What's App</a>
        <a class="btn" href="javascript:location.reload()">Retry</a>
      </div>
    </div>

    <div class="side">
      <h3>Details</h3>
      <div class="kv">
        <div>Status</div><div>403 Forbidden</div>
        <div>Host</div><div>$safeHost</div>
        <div>Date</div><div>{$dt->format('Y-m-d H:i:s')}</div>
      </div>
    </div>

  </div>

  <div class="footer">
    <div>© {$dt->format('Y')} $safeBrand</div>
    <div>Security Check • <code>$requestId</code></div>
  </div>

</div>

</body>
</html>
HTML
    );
}

/**
 * ------------------------------------------------------------------
 * 5. Write daily lock (success)
 * ------------------------------------------------------------------
 */
file_put_contents($lockFile, date('c'));
@chmod($lockFile, 0600);
